package com.cognex.cmb;

import android.Manifest;
import android.app.Activity;
import android.content.pm.PackageManager;
import android.os.Build;

import com.cognex.cmbcrossplatform.CCPBridge;
import com.cognex.cmbcrossplatform.enums.ImageSourceType;
import com.cognex.cmbcrossplatform.interfaces.APIResponseListener;
import com.cognex.cmbcrossplatform.interfaces.PermissionBridgeListener;
import com.cognex.cmbcrossplatform.interfaces.ReaderDeviceBridgeListener;

import org.apache.cordova.CallbackContext;
import org.apache.cordova.CordovaPlugin;
import org.apache.cordova.PluginResult;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * This class executes native code when called from JavaScript.
 */
public class ScannerBridge extends CordovaPlugin {

    CallbackContext didReceiveReadResultFromReaderCallbackID;
    CallbackContext availabilityDidChangeOfReaderCallbackID;
    CallbackContext connectionStateDidChangeOfReaderCallbackID;
    CallbackContext scanningStateChangedCallbackId;

    private static final int MANUAL_CAMERA_REQUEST_PERMISSION_CODE = 567;
    CallbackContext manualRequestPermissionCallbackId;

    private CCPBridge ccpBridge;
    private final ReaderDeviceBridgeListener readerDeviceBridgeListener = new ReaderDeviceBridgeListener() {
        @Override
        public void connectionStateChanged(int connectionState) {
            if (connectionStateDidChangeOfReaderCallbackID != null) {
                final PluginResult pr = new PluginResult(PluginResult.Status.OK, connectionState);
                pr.setKeepCallback(true);
                connectionStateDidChangeOfReaderCallbackID.sendPluginResult(pr);
            }
        }

        @Override
        public void readResultReceived(JSONObject result) {
            if (didReceiveReadResultFromReaderCallbackID != null) {
                try {
                    result.put("readResults", result.remove("results"));
                    result.put("subReadResults", result.remove("subResults"));
                } catch (JSONException e) {
                    e.printStackTrace();
                }

                final PluginResult pr = new PluginResult(PluginResult.Status.OK, result);
                pr.setKeepCallback(true);
                didReceiveReadResultFromReaderCallbackID.sendPluginResult(pr);
            }
        }

        @Override
        public void availabilityChanged(int availability) {
            if (availabilityDidChangeOfReaderCallbackID != null) {
                final PluginResult pr = new PluginResult(PluginResult.Status.OK, availability);
                pr.setKeepCallback(true);
                availabilityDidChangeOfReaderCallbackID.sendPluginResult(pr);
            }
        }

        @Override
        public void scanningStateChanged(boolean state) {
            if (scanningStateChangedCallbackId != null) {
                final PluginResult pr = new PluginResult(PluginResult.Status.OK, state);
                pr.setKeepCallback(true);
                scanningStateChangedCallbackId.sendPluginResult(pr);
            }
        }
    };

    private final PermissionBridgeListener permissionBridgeListener = (permissions, requestCode) -> {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            cordova.requestPermissions(this, requestCode, permissions);
        }
    };

    @Override
    public void onRequestPermissionResult(int requestCode, String[] permissions, int[] grantResults) throws JSONException {
        if (requestCode == CCPBridge.CAMERA_REQUEST_PERMISSION_CODE) {
            if (grantResults.length > 0 && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                ccpBridge.connect(null);
            }
        } else if (requestCode == MANUAL_CAMERA_REQUEST_PERMISSION_CODE) {
            checkCameraPermission(manualRequestPermissionCallbackId);
            manualRequestPermissionCallbackId = null;
        } else
            super.onRequestPermissionResult(requestCode, permissions, grantResults);
    }

    @Override
    protected void pluginInitialize() {
        super.pluginInitialize();

        ccpBridge = new CCPBridge(cordova.getActivity(), readerDeviceBridgeListener, permissionBridgeListener);
        cordova.getActivity().getApplication()
                .registerActivityLifecycleCallbacks(ccpBridge.lifecycleCallbacks);
    }

    @Override
    public void onDestroy() {
        final Activity currentActivity = cordova.getActivity();
        if (currentActivity != null) {
            currentActivity.getApplication()
                    .unregisterActivityLifecycleCallbacks(ccpBridge.lifecycleCallbacks);
        }

        ccpBridge = null;
        didReceiveReadResultFromReaderCallbackID = null;
        availabilityDidChangeOfReaderCallbackID = null;
        connectionStateDidChangeOfReaderCallbackID = null;
        scanningStateChangedCallbackId = null;

        super.onDestroy();
    }

    @Override
    public boolean execute(String action, JSONArray args, final CallbackContext callbackContext)
            throws JSONException {
        switch (action) {
            case "didReceiveReadResultFromReaderCallback":
                didReceiveReadResultFromReaderCallbackID = callbackContext;
                return true;
            case "availabilityDidChangeOfReaderCallback":
                availabilityDidChangeOfReaderCallbackID = callbackContext;
                return true;
            case "connectionStateDidChangeOfReaderCallback":
                connectionStateDidChangeOfReaderCallbackID = callbackContext;
                return true;
            case "setActiveStartScanningCallback":
                scanningStateChangedCallbackId = callbackContext;
                return true;
            case "loadScanner":
                loadScanner(callbackContext, args.getInt(0));
                return true;
            case "getAvailability":
                getAvailability(callbackContext);
                return true;
            case "enableImage":
                enableImage(callbackContext, args.getBoolean(0));
                return true;
            case "enableImageGraphics":
                enableImageGraphics(callbackContext, args.getBoolean(0));
                return true;
            case "getConnectionState":
                getConnectionState(callbackContext);
                return true;
            case "connect":
                connect(callbackContext);
                return true;
            case "startScanning":
                startScanning(callbackContext);
                return true;
            case "stopScanning":
                stopScanning(callbackContext);
                return true;
            case "getDeviceBatteryLevel":
                getDeviceBatteryLevel(callbackContext);
                return true;
            case "setSymbologyEnabled":
                setSymbologyEnabled(callbackContext, args.getInt(0), args.getBoolean(1));
                return true;
            case "isSymbologyEnabled":
                isSymbologyEnabled(callbackContext, args.getInt(0));
                return true;
            case "setLightsOn":
                setLightsOn(callbackContext, args.getBoolean(0));
                return true;
            case "isLightsOn":
                isLightsOn(callbackContext);
                return true;
            case "resetConfig":
                resetConfig(callbackContext);
                return true;
            case "sendCommand":
                sendCommand(callbackContext, args.getString(0));
                return true;
            case "setPreviewContainerPositionAndSize":
                setPreviewContainerPositionAndSize(args.getDouble(0), args.getDouble(1),
                        args.getDouble(2), args.getDouble(3));
                return true;
            case "setCameraMode":
                setCameraMode(args.getInt(0));
                return true;
            case "setPreviewOptions":
                setPreviewOptions(args.getInt(0));
                return true;
            case "setPreviewOverlayMode":
                setPreviewOverlayMode(args.getInt(0));
                return true;
            case "disconnect":
                disconnect(callbackContext);
                return true;
            case "beep":
                beep(callbackContext);
                return true;
            case "registerSDK":
                registerSDK(args.getString(0));
                return true;
            case "enableCameraFlag":
                editCameraFlag(callbackContext, args.getInt(0), args.getInt(1), true);
                return true;
            case "disableCameraFlag":
                editCameraFlag(callbackContext, args.getInt(0), args.getInt(1), false);
                return true;
            case "setCameraParam":
                setCameraParam(callbackContext, args.getInt(0), args.getInt(1), args.getInt(2));
                return true;
            case "setCameraDuplicatesTimeout":
                setCameraDuplicatesTimeout(callbackContext, args.getInt(0));
                return true;
            case "showToast":
                showToast(args.getString(0));
                return true;
            case "hideToast":
                hideToast();
                return true;
            case "getSdkVersion":
                getSdkVersion(callbackContext);
                return true;
            case "checkCameraPermission":
                checkCameraPermission(callbackContext);
                return true;
            case "requestCameraPermission":
                requestCameraPermission(callbackContext);
                return true;
            case "setPreviewContainerFullScreen":
                setPreviewContainerFullScreen(callbackContext);
                return true;
            case "setParser":
                setParser(callbackContext, args.getInt(0));
                return true;
            case "setReadStringEncoding":
                setReadStringEncoding(callbackContext, args.getInt(0));
                return true;
            case "setStopScannerOnRotate":
                setStopScannerOnRotate(args.getBoolean(0));
                return true;
            case "scanImageFromUri":
                scanImage(callbackContext, ImageSourceType.URI, args.getString(0));
                return true;
            case "scanImageFromBase64":
                scanImage(callbackContext, ImageSourceType.BASE64, args.getString(0));
                return true;
            case "getCameraExposureCompensationRange":
                getCameraExposureCompensationRange(callbackContext);
                return true;
            case "setCameraExposureCompensation":
                setCameraExposureCompensation(callbackContext, args.getDouble(0));
                return true;
            case "loadCameraConfig":
                loadCameraConfig(callbackContext);
                return true;
            case "setPairedBluetoothDevice":
                setPairedBluetoothDevice(args.getString(0));
                return true;
            default:
                return false;
        }
    }

    private APIResponseListener createResponseListener(final CallbackContext callbackContext) {
        return new APIResponseListener() {
            @Override
            public void successWithObject(Object result) {
                final PluginResult pr;
                if (result == null) {
                    pr = new PluginResult(PluginResult.Status.OK);
                } else if (result instanceof Integer) {
                    pr = new PluginResult(PluginResult.Status.OK, (int) result);
                } else if (result instanceof Boolean) {
                    pr = new PluginResult(PluginResult.Status.OK, (boolean) result);
                } else if (result instanceof Float) {
                    pr = new PluginResult(PluginResult.Status.OK, (float) result);
                } else if (result instanceof JSONArray) {
                    pr = new PluginResult(PluginResult.Status.OK, (JSONArray) result);
                } else if (result instanceof JSONObject) {
                    pr = new PluginResult(PluginResult.Status.OK, (JSONObject) result);
                } else {
                    pr = new PluginResult(PluginResult.Status.OK, result.toString());
                }

                callbackContext.sendPluginResult(pr);
            }

            @Override
            public void error(String errorCode, String errorMessage) {
                final JSONObject jsonError = new JSONObject();
                try {
                    jsonError.put("errorCode", errorCode);
                    jsonError.put("errorMessage", errorMessage);
                } catch (JSONException e) {
                    e.printStackTrace();
                }

                callbackContext.error(jsonError);
            }
        };
    }

    //region CCPBridge methods

    private void loadScanner(final CallbackContext callbackContext, int deviceType) {
        ccpBridge.loadScanner(createResponseListener(callbackContext), deviceType);
    }

    private void connect(final CallbackContext callbackContext) {
        ccpBridge.connect(createResponseListener(callbackContext));
    }

    private void disconnect(final CallbackContext callbackContext) {
        ccpBridge.disconnect(createResponseListener(callbackContext));
    }

    private void startScanning(final CallbackContext callbackContext) {
        ccpBridge.startScanning(createResponseListener(callbackContext));
    }

    private void stopScanning(final CallbackContext callbackContext) {
        ccpBridge.stopScanning(createResponseListener(callbackContext));
    }

    private void setSymbologyEnabled(final CallbackContext callbackContext, int symbology, boolean enable) {
        ccpBridge.setSymbologyEnabled(createResponseListener(callbackContext), symbology, enable);
    }

    private void isSymbologyEnabled(final CallbackContext callbackContext, int symbology) {
        ccpBridge.isSymbologyEnabled(createResponseListener(callbackContext), symbology);
    }

    private void setCameraMode(int cameraMode) {
        ccpBridge.setCameraMode(cameraMode);
    }

    private void setPreviewOptions(int previewOptions) {
        ccpBridge.setPreviewOptions(previewOptions);
    }

    private void setPreviewOverlayMode(int overlayMode) {
        ccpBridge.setPreviewOverlayMode(overlayMode);
    }

    private void setPreviewContainerPositionAndSize(double x, double y, double width, double height) {
        ccpBridge.setPreviewContainerPositionAndSize(x, y, width, height);
    }

    private void setLightsOn(CallbackContext callbackContext, boolean on) {
        ccpBridge.setLightsOn(createResponseListener(callbackContext), on);
    }

    private void isLightsOn(CallbackContext callbackContext) {
        ccpBridge.isLightsOn(createResponseListener(callbackContext));
    }

    private void enableImage(CallbackContext callbackContext, boolean enable) {
        ccpBridge.enableImage(createResponseListener(callbackContext), enable);
    }

    private void enableImageGraphics(CallbackContext callbackContext, boolean enable) {
        ccpBridge.enableImageGraphics(createResponseListener(callbackContext), enable);
    }

    private void sendCommand(CallbackContext callbackContext, String commandString) {
        ccpBridge.sendCommand(createResponseListener(callbackContext), commandString);
    }

    private void resetConfig(CallbackContext callbackContext) {
        ccpBridge.resetConfig(createResponseListener(callbackContext));
    }

    private void registerSDK(String key) {
        ccpBridge.registerSDK(key);
    }

    private void getAvailability(CallbackContext callbackContext) {
        ccpBridge.getAvailability(createResponseListener(callbackContext));
    }

    private void getConnectionState(CallbackContext callbackContext) {
        ccpBridge.getConnectionState(createResponseListener(callbackContext));
    }

    private void beep(CallbackContext callbackContext) {
        ccpBridge.beep(createResponseListener(callbackContext));
    }

    private void getDeviceBatteryLevel(CallbackContext callbackContext) {
        ccpBridge.getDeviceBatteryLevel(createResponseListener(callbackContext));
    }

    private void getSdkVersion(CallbackContext callbackContext) {
        ccpBridge.getSdkVersion(createResponseListener(callbackContext));
    }

    private void setPreviewContainerFullScreen(CallbackContext callbackContext) {
        ccpBridge.setPreviewContainerFullScreen(createResponseListener(callbackContext));
    }

    private void setStopScannerOnRotate(boolean stop) {
        ccpBridge.setStopScannerOnRotate(stop);
    }

    private void scanImage(CallbackContext callbackContext, ImageSourceType sourceType, String source) {
        ccpBridge.scanImage(createResponseListener(callbackContext), sourceType, source);
    }

    private void editCameraFlag(CallbackContext callbackContext, int mask, int flag, boolean enable) {
        ccpBridge.editCameraFlag(createResponseListener(callbackContext), mask, flag, enable);
    }

    private void setCameraParam(CallbackContext callbackContext, int mask, int param, int value) {
        ccpBridge.setCameraParam(createResponseListener(callbackContext), param, value, mask);
    }

    private void setParser(CallbackContext callbackContext, int parser) {
        ccpBridge.setParser(createResponseListener(callbackContext), parser);
    }

    private void setReadStringEncoding(CallbackContext callbackContext, int encoding) {
        ccpBridge.setReadStringEncoding(createResponseListener(callbackContext), encoding);
    }

    private void getCameraExposureCompensationRange(CallbackContext callbackContext) {
        ccpBridge.getCameraExposureCompensationRange(createResponseListener(callbackContext));
    }

    private void setCameraExposureCompensation(CallbackContext callbackContext, double exposureCompensation) {
        ccpBridge.setCameraExposureCompensation(createResponseListener(callbackContext), (float) exposureCompensation);
    }

    private void loadCameraConfig(CallbackContext callbackContext) {
        ccpBridge.loadCameraConfig(createResponseListener(callbackContext));
    }

    private void setPairedBluetoothDevice(String btDevice) {
        ccpBridge.setPairedBTDeviceMacAddress(btDevice);
    }

    private void showToast(String message) {
        ccpBridge.showToast(message);
    }

    private void hideToast() {
        ccpBridge.hideToast(true);
    }

    //endregion CCPBridge methods

    //region Cordova specific methods

    private void setCameraDuplicatesTimeout(CallbackContext callbackContext, int timeoutInS) {
        ccpBridge.sendCommand(createResponseListener(callbackContext),
                "SET DECODER.REREAD-TIME " + timeoutInS * 1000);
    }

    private void checkCameraPermission(CallbackContext callbackContext) {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M || cordova.hasPermission(Manifest.permission.CAMERA)) {
            callbackContext.success();
        } else {
            try {
                final Activity currentActivity = cordova.getActivity();
                if (currentActivity != null && currentActivity.shouldShowRequestPermissionRationale(Manifest.permission.CAMERA)) {
                    callbackContext.error(1);
                } else {
                    callbackContext.error(0);
                }
            } catch (Exception e) {
                callbackContext.error(e.getMessage());
            }
        }
    }

    private void requestCameraPermission(CallbackContext callbackContext) {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M || cordova.hasPermission(Manifest.permission.CAMERA)) {
            callbackContext.success();
        } else {
            try {
                manualRequestPermissionCallbackId = callbackContext;
                cordova.requestPermission(ScannerBridge.this,
                        MANUAL_CAMERA_REQUEST_PERMISSION_CODE, Manifest.permission.CAMERA);
            } catch (Exception e) {
                callbackContext.error(e.getMessage());
            }
        }
    }

    //endregion Cordova specific methods
}
