import { Observable, ObservableArray, ApplicationEventData, Application, Dialogs } from '@nativescript/core';
import { CMBReader, CMBReaderConstants } from 'cmbsdk-nativescript';

class ResultListViewItem {
  text: string;
  symbology: string;

  constructor(_text: string, _symbology: string) {
    this.text = _text;
    this.symbology = _symbology;
  }
}

export class HomeViewModel extends Observable {

  private cmb: CMBReader;
  private isScanning: boolean = false;

  //----------------------------------------------------------------------------
  // The cmbSDK supports multi-code scanning (scanning multiple barcodes at
  // one time); thus scan results are returned as an array. Note that
  // this sample app does not demonstrate the use of these multi-code features.
  //----------------------------------------------------------------------------
  private listViewArray: ObservableArray<ResultListViewItem>;
  private scanButtonText: string;
  private scanButtonEnabled: boolean;
  private cmbsdkVersionText: string;
  private connectionStatusText: string;
  private connectionStatusBackground: string;

  //----------------------------------------------------------------------------
  // If USE_PRECONFIGURED_DEVICE is YES, then the app will create a reader
  // using the values of device/cameraMode. Otherwise, the app presents
  // a pick list for the user to select either MX-1xxx or the built-in camera.
  //----------------------------------------------------------------------------
  private readonly USE_PRECONFIGURED_DEVICE: boolean = false;
  private deviceType: number = CMBReaderConstants.DEVICE_TYPE.MXReader;
  private cameraMode: number = CMBReaderConstants.CAMERA_MODE.NoAimer;

  constructor() {
    super();

    this.listViewArray = new ObservableArray<ResultListViewItem>();
    this.scanButtonText = "START SCAN";
    this.scanButtonEnabled = false;
    this.cmbsdkVersionText = "N/A";
    this.connectionStatusText = "Disconnected";
    this.connectionStatusBackground = "#ff4444";

    this.cmb = new CMBReader();

    // This is called when a MX-1xxx device has became available (USB cable was plugged, or MX device was turned on),
    // or when a MX-1xxx that was previously available has become unavailable (USB cable was unplugged, turned off due to inactivity or battery drained)
    this.cmb.on(CMBReaderConstants.EVENTS.AvailabilityChanged, (args: any) => {
      if (args.data == CMBReaderConstants.AVAILABILITY.Available) {
        this.cmb.connect();
      } else {
        Dialogs.alert({
          message: "Device became unavailable",
          okButtonText: "OK"
        });
      }
    });

    // This is called when a connection with the reader has been changed.
    // The reader is usable only in the "Connected" state
    this.cmb.on(CMBReaderConstants.EVENTS.ConnectionStateChanged, (args: any) => {
      this.listViewArray.splice(0);

      if (args.data == CMBReaderConstants.CONNECTION_STATE.Connected) {
        // We just connected, so now configure the reader how we want it
        this.configureReaderDevice();
      }

      this.isScanning = false;
      this.updateUIByConnectionState(args.data);
    });

    //It will return TRUE in the result if the scanning process is STARTED and false if it's STOPPED
    this.cmb.on(CMBReaderConstants.EVENTS.ScanningStateChanged, (args: any) => {
      this.isScanning = args.data;
      this.set('scanButtonText', this.isScanning ? "STOP SCAN" : "START SCAN");
    });

    // This is called after scanning has completed, either by detecting a barcode,
    // canceling the scan by using the on-screen button or a hardware trigger button, or if the scanning timed-out
    this.cmb.on(CMBReaderConstants.EVENTS.ReadResultReceived, (args: any) => {
      this.listViewArray.splice(0);

      if (args.data.subResults != null && args.data.subResults.length > 0) {
        args.data.subResults.forEach(element => {
          this.listViewArray.push(new ResultListViewItem(element.readString, element.symbologyString));
        });
      } else if (args.data.results.length > 0) {
        this.listViewArray.push(new ResultListViewItem(args.data.results[0].readString, args.data.results[0].symbologyString));
      }
    });

    // Get cmbSDK version number
    this.cmbsdkVersionText = this.cmb.getSdkVersion();

    //initialize and connect to MX/Phone Camera here
    if (this.USE_PRECONFIGURED_DEVICE) {
      this.createReaderDevice();
    } else {
      this.selectDeviceFromPicker();
    }

    //Clear already initialized functions for this event to prevent event pooling
    Application.off(Application.suspendEvent);
    Application.on(Application.suspendEvent, (args: ApplicationEventData) => {
      if (this.cmb != null) {
        this.cmb.disconnect();
      }
    });

    //Clear already initialized functions for this event to prevent event pooling
    Application.off(Application.resumeEvent);
    Application.on(Application.resumeEvent, (args: ApplicationEventData) => {
      if (this.cmb != null) {
        this.cmb.connect();
      }
    });

    //Clear already initialized functions for this event to prevent event pooling
    Application.off(Application.exitEvent);
    Application.on(Application.exitEvent, (args: ApplicationEventData) => {
      this.cmb = null;
      Application.off(Application.suspendEvent);
      Application.off(Application.resumeEvent);
      Application.off(Application.exitEvent);
    });
  }

  // Update the UI of the app (scan button, connection state label) depending on the current connection state
  private updateUIByConnectionState(currentConnectionState: any): void {
    if (currentConnectionState == CMBReaderConstants.CONNECTION_STATE.Connected) {
      this.set('connectionStatusText', "Connected");
      this.set('connectionStatusBackground', "#669900");

      this.set('scanButtonEnabled', true);
    } else {
      this.set('connectionStatusText', "Disconnected");
      this.set('connectionStatusBackground', "#ff4444");

      this.set('scanButtonEnabled', false);
    }

    this.set('scanButtonText', this.scanButtonEnabled ? "START SCANNING" : "(NOT CONNECTED)");
  }

  //----------------------------------------------------------------------------
  // This is the pick list for choosing the type of reader connection
  //----------------------------------------------------------------------------
  private selectDeviceFromPicker(): void {
    let actionOptions = ["MX Scanner (MX-1xxx)", "Phone Camera"];

    Dialogs.action({
      message: "Select device",
      cancelButtonText: "Cancel",
      actions: actionOptions
    }).then(result => {
      if (result == "MX Scanner (MX-1xxx)") {
        this.deviceType = CMBReaderConstants.DEVICE_TYPE.MXReader;
      } else if (result == "Phone Camera") {
        this.deviceType = CMBReaderConstants.DEVICE_TYPE.Camera;
        this.cameraMode = CMBReaderConstants.CAMERA_MODE.NoAimer;
      }

      this.createReaderDevice();
    });
  }

  // Create a reader using the selected option from "selectDeviceFromPicker"
  private createReaderDevice(): void {

    this.cmb.setCameraMode(this.cameraMode);
    this.cmb.setPreviewOptions(CMBReaderConstants.CAMERA_PREVIEW_OPTION.Defaults);
    this.cmb.loadScanner(this.deviceType)
      .then(result => {
        this.cmb.connect()
          .then(result => {
            this.updateUIByConnectionState(CMBReaderConstants.CONNECTION_STATE.Connected);
          })
          .catch(err => {
            console.log(err);
            this.updateUIByConnectionState(CMBReaderConstants.CONNECTION_STATE.Disconnected);
          });
      });
  }

  //----------------------------------------------------------------------------
  // This is an example of configuring the device. In this sample application, we
  // configure the device every time the connection state changes to connected (see
  // the ConnectionStateChanged event), as this is the best
  // way to garentee it is setup the way we want it. Not only does this garentee
  // that the device is configured when we initially connect, but also covers the
  // case where an MX scanner has hibernated (and we're reconnecting)--unless
  // setting changes are explicitly saved to non-volatile memory, they can be lost
  // when the MX hibernates or reboots.
  //
  // These are just example settings; in your own application you will want to
  // consider which setting changes are optimal for your application. It is
  // important to note that the different supported devices have different, out
  // of the box defaults:
  //
  //    * MX-1xxx Mobile Terminals have the following symbologies enabled by default:
  //        - Data Matrix
  //        - UPC/EAN
  //        - Code 39
  //        - Code 93
  //        - Code 128
  //        - Interleaved 2 of 5
  //        - Codabar
  //    * camera scanner has NO symbologies enabled by default
  //
  // For the best scanning performance, it is recommended to only have the barcode
  // symbologies enabled that your application actually needs to scan. If scanning
  // with an MX-1xxx, that may mean disabling some of the defaults (or enabling
  // symbologies that are off by default).
  //
  // Keep in mind that this sample application works with all three types of devices,
  // so in our example below we show explicitly enabling symbologies as well as
  // explicitly disabling symbologies (even if those symbologies may already be on/off
  // for the device being used).
  //
  // We also show how to send configuration commands that may be device type
  // specific--again, primarily for demonstration purposes.
  //----------------------------------------------------------------------------
  private configureReaderDevice(): void {
    //----------------------------------------------
    // Explicitly enable the symbologies we need
    //----------------------------------------------
    this.cmb.setSymbologyEnabled(CMBReaderConstants.SYMBOLOGY.DataMatrix, true)
      .then(function (result) {
        if (result == true)
          console.log("DataMatrix enabled");
        else
          console.log("DataMatrix NOT enabled");
      })
      .catch(function (err) {
        console.log(err);
      });

    this.cmb.setSymbologyEnabled(CMBReaderConstants.SYMBOLOGY.QR, true);
    this.cmb.setSymbologyEnabled(CMBReaderConstants.SYMBOLOGY.UpcEan, true);

    //-------------------------------------------------------
    // Explicitly disable symbologies we know we don't need
    //-------------------------------------------------------
    this.cmb.setSymbologyEnabled(CMBReaderConstants.SYMBOLOGY.CodaBar, false)
      .then(function (result) {
        if (result == true)
          console.log("CodaBar disabled");
        else
          console.log("CodaBar NOT disabled");
      })
      .catch(function (err) {
        console.log(err);
      });

    this.cmb.setSymbologyEnabled(CMBReaderConstants.SYMBOLOGY.C93, false);

    //---------------------------------------------------------------------------
    // Below are examples of sending DMCC commands and getting the response
    //---------------------------------------------------------------------------
    this.cmb.sendCommand("GET DEVICE.TYPE")
      .then(function (result) {
        console.log(result);
      })
      .catch(function (result) {
        console.log(result);
      });

    this.cmb.sendCommand("GET DEVICE.FIRMWARE-VER")
      .then(function (result) {
        console.log(result);
      })
      .catch(function (result) {
        console.log(result);
      });

    //---------------------------------------------------------------------------
    // We are going to explicitly turn off image results (although this is the
    // default). The reason is that enabling image results with an MX-1xxx
    // scanner is not recommended unless your application needs the scanned
    // image--otherwise scanning performance can be impacted.
    //---------------------------------------------------------------------------
    this.cmb.enableImage(false);
    this.cmb.enableImageGraphics(false);

    //---------------------------------------------------------------------------
    // Device specific configuration examples
    //---------------------------------------------------------------------------
    if (this.deviceType == CMBReaderConstants.DEVICE_TYPE.Camera) {
      //---------------------------------------------------------------------------
      // Phone/tablet
      //---------------------------------------------------------------------------

      // Set the SDK's decoding effort to level 3
      this.cmb.sendCommand("SET DECODER.EFFORT 3");
    } else if (this.deviceType == CMBReaderConstants.DEVICE_TYPE.MXReader) {
      //---------------------------------------------------------------------------
      // MX-1xxx
      //---------------------------------------------------------------------------

      //---------------------------------------------------------------------------
      // Save our configuration to non-volatile memory
      // If the MX hibernates or is
      // rebooted, our settings will be retained.
      //---------------------------------------------------------------------------
      this.cmb.sendCommand("CONFIG.SAVE");
    }
  }

  public toggleScanner(): void {
    if (this.isScanning) {
      this.cmb.stopScanning()
        .catch(error => { console.log(error); });
    } else {
      this.cmb.startScanning()
        .catch(error => { console.log(error); });
    }
  }
}
