//
//  ScannerView.swift
//  cmbSDKToolkitSample
//
//

import SwiftUI
import cmbSDK
import Combine

struct ScannerView: View {
    
    @EnvironmentObject var appState: AppState
    @ObservedObject private var readerState: ReaderState
    @State private var settingsIsPresented: Bool = false
    
    init(_ readerState: ReaderState) {
        self.readerState = readerState
    }
    
    var body: some View {
        
        NavigationStack {
            VStack {
                switch readerState.connectionState {
                    case CMBConnectionStateDisconnected:
                        Text("Reader is disconnected.")
                            .padding()
                    case CMBConnectionStateConnecting:
                        Text("Connecting...")
                            .padding()
                        ProgressView()
                            .padding()
                            .scaleEffect(2)
                    case CMBConnectionStateConnected:
                    if let readResult = readerState.readResult {
                        Text(self.displayStringForSymbology(readResult.symbology))
                            .padding()
                        Text(readResult.readString ?? "")
                            .padding()
                    } else {
                        Text("Scan something to see the result here.")
                            .padding()
                    }
                    
                    default:
                        Text("")
                }
                
                if let image = readerState.readResult?.image, appState.readerState.readerConfig.showImage {
                    Image(uiImage: image)
                        .imageScale(.large)
                        .foregroundColor(.accentColor)
                        .scaledToFit()
                        .aspectRatio(contentMode: .fit)
                }
            }
            .toolbar {
                if readerState.connectionState == CMBConnectionStateConnected {
                    Button {
                        settingsIsPresented = true
                    } label: {
                        Image(systemName: "gearshape")
                    }
                }
            }
        }
        .sheet(isPresented: $settingsIsPresented) {
            
        } content: {
            SettingsView(readerState.readerConfig)
        }

    }
    
    // Get a readable string from a CMBSymbology value
    func displayStringForSymbology(_ symbology: CMBSymbology) -> String
    {
        switch symbology {
        case CMBSymbologyDataMatrix: return "DATAMATRIX";
        case CMBSymbologyQR: return "QR";
        case CMBSymbologyC128: return "C128";
        case CMBSymbologyUpcEan: return "UPC-EAN";
        case CMBSymbologyC39: return "C39";
        case CMBSymbologyC93: return "C93";
        case CMBSymbologyC11: return "C11";
        case CMBSymbologyI2o5: return "I2O5";
        case CMBSymbologyCodaBar: return "CODABAR";
        case CMBSymbologyEanUcc: return "EAN-UCC";
        case CMBSymbologyPharmaCode: return "PHARMACODE";
        case CMBSymbologyMaxicode: return "MAXICODE";
        case CMBSymbologyPdf417: return "PDF417";
        case CMBSymbologyMicropdf417: return "MICROPDF417";
        case CMBSymbologyDatabar: return "DATABAR";
        case CMBSymbologyPostnet: return "POSTNET";
        case CMBSymbologyPlanet: return "PLANET";
        case CMBSymbologyFourStateJap: return "4STATE-JAP";
        case CMBSymbologyFourStateAus: return "4STATE-AUS";
        case CMBSymbologyFourStateUpu: return "4STATE-UPU";
        case CMBSymbologyFourStateImb: return "4STATE-IMB";
        case CMBSymbologyVericode:return "VERICODE";
        case CMBSymbologyRpc: return "RPC";
        case CMBSymbologyMsi: return "MSI";
        case CMBSymbologyAzteccode: return "AZTECCODE";
        case CMBSymbologyDotcode: return "DOTCODE";
        case CMBSymbologyC25: return "C25";
        case CMBSymbologyC39ConvertToC32: return "C39-CONVERT-TO-C32";
        case CMBSymbologyOcr: return "OCR";
        case CMBSymbologyFourStateRmc: return "4STATE-RMC";
        case CMBSymbologyTelepen: return "TELEPEN";
        default: return "UNKNOWN";
        }
    }
}
